unit PatternmatchThread;

interface

uses   SysUtils, Classes, Graphics;


const  SA_NAIV = 0;
       SA_KMP_F = 1;
       SA_KMP_NEXT = 2;
       SA_BM = 3;
       SA_BM_Weak = 4;
       SA_HORSPOOL = 5;

       ANIMATE_CHARWIDTH = 25;
       ANIMATE_FONTZIZE = 12;


type
  TIndexArray = Array of Integer;

  // Prozedur, die ein neuzeichnen im VCL-Thread erwirken soll
  // ber den Parameter "Mode" entscheidet die VCL-Methode, wohin das Bitmap geziechnet werden soll
  TRepaintProc =  //procedure(Mode: Integer; aBitmap: TBitmap) of Object;
      procedure(Mode, StartPos, StartG, EndG, StartR, EndR, Gelb: Integer; aShift: Integer = 0) of Object;

  TPatternmatchAnimator = class(TThread)
  private
        { Private-Deklarationen }
        fText: String;
        fPattern: String;

        // Zhlvariable fr die Anzahl der Vergleiche
        fCount: Int64;
        // Indikator fr den "Suchmodus", d.h. welcher Algorithmus verwendet werden soll
        fMode: Integer;
        // *** Preprocessing - Arrays ***
        // Next-Array fr den Algorithmus nach Knuth-Morris-Pratt
        fKMP_Next: TIndexArray;
        //"F-Array" fr den Algorithmus nach Knuth-Morris-Pratt.
        // Manchmal wird auch dieses als Next bezeichnet
        fKMP_F: TIndexArray;
        // Good-Suffix-Shift fr Boyer-Moore
        fBM_GS: TIndexArray;
        //Bad-Character-Shift fr Boyer-Moore
        fBM_BC: TIndexArray;

        // Position des Musters im Text nach einer Suche
        FPatternFoundAt: Integer;



        fDelayTime: Integer;
        fMachePause: Boolean;

        fRepaintPatternProc: TRepaintProc;
        fRepaintTextProc: TRepaintProc;

        // Startposition im Text
        // d.h. linke Position des "Suchfensters"
        fStartPos     : Integer;

        // Hilfsvariable frs Shiften
        fShift        : Integer;
        //Markierungen fr aktuell bekannte Match/Mismatches im Text
        fStartGText       : Integer;
        fEndGText         : Integer;
        fStartRText       : Integer;
        fEndRText         : Integer;
        fGelbText         : Integer;
        //Markierungen fr aktuell bekannte Match/Mismatches im Pattern
        fStartGPattern       : Integer;
        fEndGPattern         : Integer;
        fStartRPattern       : Integer;
        fEndRPattern         : Integer;
        fGelbPattern         : Integer;

        procedure SetPatternMatch(Start, Ende: Integer);
        procedure SetPatternMismatch(Start, Ende: Integer);
        procedure SetPatternNextTest(Start: Integer);
        procedure SetTextMatch(Start, Ende: Integer);
        procedure SetTextMismatch(Start, Ende: Integer);
        procedure SetTextNextTest(Start: Integer);

        // *** Funktionen frs Preprocessing ***
        // Berechnung des Next/F-Arrays fr den KMP-Algorithmus.
        // D.E.Knuth, J.H.Morris und V.R.Pratt: Fast pattern matching in strings
        // SIAM Journal on Computing 6(2), 1977 pp 323-350
        procedure PreProcess_KMP_Next;
        procedure PreProcess_KMP_F;
        // Berechnung des _strong_ Good-Suffix-Arrays fr den Boyer-Moore-Algorithmus
        // Code nach Cole aus:
        // D.Gusfield: Algorithms on Strings, Trees, and Sequences
        // Cambridge University Press, 1997
        // Exercise 24 in Chapter I.2, pages 31-33
        procedure PreProcess_BM_GSCole;
        // Berechnung des Bad-Character-Shifts
        procedure PreProcess_BM_BadCharacter;
        Procedure PreProcess_BM_Complete;

        // *** Suchfunktionen ***
        // Naiver Algorithmus
        function Search_Naiv_Count: Integer;
        // Knuth-Morris-Pratt
        function Search_KMP_Count(aShiftArray: TIndexArray): Integer;
        // Knuth-Morris-Pratt in "schlechterer Version"
        // Boyer-Moore
        function Search_BM_Count: Integer;
        function Search_Horspool_Count: Integer;

        procedure RefreshPattern;
        procedure RefreshText;
        procedure ShiftTextNew;
        Procedure SleepAWhile(aWhile: Integer);


  protected
        procedure Execute; override;

  public
        UseAnimation: Boolean;
        // eine Kopie des gesamten zu malenden Bereichs

        property Text: String read fText write fText;
        property Pattern: String read fPattern write fPattern;
        property Count: Int64 read fCount;
        property PatternFoundAt: Integer read FPatternFoundAt;
        property Mode: Integer read fMode write fMode;

        property DelayTime: Integer read fDelayTime write fDelayTime;
        property MachePause: Boolean read fMachePause write fMachePause;

        property RepaintPatternProc: TRepaintProc read fRepaintPatternProc write fRepaintPatternProc;
        property RepaintTextProc: TRepaintProc read fRepaintTextProc write fRepaintTextProc;



        constructor Create(CreateSuspended: Boolean);
        destructor Destroy; override;


        procedure PreProcessPattern;
        function Search_Count: Integer;
  end;

  function max(a,b: Integer): Integer;
  function GenerateRandomString(aLength: Integer; Sigma: Integer): String;

implementation

function max(a,b: Integer): Integer;
begin
  if a > b then
    result := a
  else
    result := b;
end;

function GenerateRandomString(aLength: Integer; Sigma: Integer): String;
var i: Integer;
begin
  setlength(result, aLength);
  for i := 1 to length(result) do
    result[i] := Chr(Random(Sigma) + ord('A'));
end;


{ Wichtig: Methoden und Eigenschaften von Objekten in visuellen Komponenten drfen 
  nur in einer Methode namens Synchronize aufgerufen werden, z.B.

      Synchronize(UpdateCaption);

  und UpdateCaption knnte folgendermaen aussehen:

    procedure TPatternmatchAnimator.UpdateCaption;
    begin
      Form1.Caption := 'Aktualisiert in einem Thread';
    end; }

{ TPatternmatchAnimator }

procedure TPatternmatchAnimator.Execute;
begin
  FPatternFoundAt := Search_Count;
end;


constructor TPatternMatchAnimator.Create(CreateSuspended: Boolean);
begin
    inherited Create(CreateSuspended);
    fText := '';
    fPattern := '';

    fDelayTime := 50;// 400;

    fStartPos := 1;

    SetLength(fBM_BC, 256);
    fCount := 0;
    FPatternFoundAt := -1;
end;

destructor TPatternMatchAnimator.Destroy;
begin
  inherited Destroy;
end;

// Muster-Preprocessing
{
  Berechnung des Next/F-Arrays fr den KMP-Algorithmus.
  D.E.Knuth, J.H.Morris und V.R.Pratt: Fast pattern matching in strings
  SIAM Journal on Computing 6(2), 1977 pp 323-350
}
procedure TPatternMatchAnimator.PreProcess_KMP_Next;
var j, t, m: Integer;
begin
  m := Length(fPattern);
  SetLength(fKMP_Next, m+1);
  j := 1;
  t := 0;
  fKMP_Next[1] := 0;
  while j < m do
  begin
    while (t > 0) AND (fPattern[j] <> fPattern[t]) do t := fKMP_Next[t];
    t := t+1;
    j := j+1;
    if fPattern[j] = fPattern[t] then
      fKMP_Next[j] := fKMP_Next[t]
    else
      fKMP_Next[j] := t;
  end;
end;
procedure TPatternMatchAnimator.PreProcess_KMP_F;
var j, t, m: Integer;
begin
  m := Length(fPattern);
  SetLength(fKMP_F, m+1);
  j := 1;
  t := 0;
  fKMP_F[1] := 0;
  while j < m do
  begin
    while (t > 0) AND (fPattern[j] <> fPattern[t]) do t := fKMP_F[t];
    t := t+1;
    j := j+1;
    fKMP_F[j] := t;
  end;
end;

{
  Berechnung des _strong_ Good-Suffix-Arrays fr den Boyer-Moore-Algorithmus
  Code nach Cole aus:
  D.Gusfield: Algorithms on Strings, Trees, and Sequences
  Cambridge University Press, 1997
  Exercise 24 in Chapter I.2, pages 31-33
}
procedure TPatternMatchAnimator.PreProcess_BM_GSCole;
var m, i, j, j_old, k: Integer;
  kmp_Shift: TIndexArray;
  go_on: boolean;
begin
  m := Length(fPattern);
  SetLength(fBM_GS, m+1);
  for j := 1 to m do
    fBM_GS[j] := m;
  SetLength(kmp_shift, m+1);
  kmp_Shift[m] := 1;

  // Stage 1
  j := m;
  for k := m-1 downto 1 do
  begin
    go_on := true;

    while (fPattern[j] <> fPattern[k]) and go_on do
    begin
      if (fBM_GS[j] > j-k) then
        fBM_GS[j] := j-k;

      if (j < m) then
        j := j + kmp_shift[j+1]
      else
        go_on := false;
    end;

    // schwache GS-Regel 
    if Mode = SA_BM_Weak then
    begin
      if fBM_GS[j] > j-k then
        fBM_GS[j] := j-k;
    end;
    // -----------

    if (fPattern[k] = fPattern[j]) then
    begin
      kmp_shift[k] := j-k;
      j := j-1;
    end else
      kmp_shift[k] := j-k+1;
  end;

  // Stage 2
  j := j+1;
  j_old := 1;

  while (j <= m) do
  begin
    for i := j_old to j-1 do
      if fBM_GS[i] > j-1 then fBM_GS[i] := j-1;
    j_old := j;
    j := j + kmp_shift[j];
  end;
end;

// Berechnung des Bad-Character-Shifts
procedure TPatternMatchAnimator.PreProcess_BM_BadCharacter;
var i, m: Integer;
begin
  m := Length(fPattern);
  for i := 0 to Length(fBM_BC)-1 do
    fBM_BC[i] := m;

  for i := 1 to m do
    fBM_BC[Ord(fPattern[i])] := m-i;
end;

Procedure TPatternMatchAnimator.PreProcess_BM_Complete;
begin
  PreProcess_BM_GSCole;
  PreProcess_BM_BadCharacter;
end;

procedure TPatternMatchAnimator.PreProcessPattern;
begin
  case fMode of
    SA_NAIV     : ;
    SA_KMP_F    : PreProcess_KMP_F;
    SA_KMP_NEXT : PreProcess_KMP_Next;
    SA_BM,SA_BM_Weak       : PreProcess_BM_Complete;
    SA_HORSPOOL : PreProcess_BM_BadCharacter;
  end;
end;


{
  Suche mit dem naiven Algorithmus mit zhlen der Char-Vergleiche
}
function TPatternMatchAnimator.Search_Naiv_Count: Integer;
var t, p, m, n: Integer;
begin
  result := -1;
  fCount := 0;
  m := Length(fPattern);
  n := Length(fText);
  for t := 1 to n - m + 1 do
  begin
    p := 1;
    if UseAnimation then
    begin
      fStartPos := t;
      fShift := 0;
    end;

    while (p <= m) and (fText[t + p - 1] = fPattern[p]) And (Not Terminated) do
    begin
      if UseAnimation then
      begin
        SetTextMatch(t, t+p-2);
        SetPatternMatch(1,p-1);
        SetTextMismatch(-1,-1);
        SetPatternMismatch(-1,-1);
        // zunchst: Den Test animieren (d.h. aktuelle Vergelichspositionen gelb)
        SetPatternNextTest(p);
        SetTextNextTest(t+p-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);

        // Jetzt: An der Vergleichsposition Grn zeichnen, das ist ja das Ergebnis des Vergleichs
        SetTextMatch(t, t+p-1);
        SetPatternMatch(1,p);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
      inc(p);
      inc(fCount)
    end;
    if  (p <= m) and (fText[t + p - 1] <> fPattern[p]) And (Not Terminated)
    then
    begin
      if UseAnimation then
      begin
        SetTextMatch(t, t+p-2);
        SetPatternMatch(1,p-1);
        SetTextMismatch(-1,-1);
        SetPatternMismatch(-1,-1);
        // zunchst: Die aktuellen Vergleichspositionen wirklich gelb zeichnen
        SetPatternNextTest(p);
        SetTextNextTest(t+p-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);

        // Jetzt: An der Vergleichsposition Rot zeichnen, das ist ja das Ergebnis des Vergleichs
        SetTextMismatch(t+p-1,t+p-1);
        SetPatternMismatch(p,p);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);

        // Mismatch gefunden - Reaktion darauf animieren
        // Dazu zunchst das Muster/Text neu malen mit den Infos, was man schon gematcht hat/nicht hat.
        // Beim naiven Algorithmus wissen wir in der Hinsicht nichts
        SetTextMatch(-1, -1);
        SetPatternMatch(-1, -1);
        SetTextMismatch(-1,-1);
        SetPatternMismatch(-1,-1);
        SetPatternNextTest(1);
        SetTextNextTest(t+1);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        // Shift starten
        ShiftTextNew;
      end;
      inc(fCount); // fr den fehlgeschlagenen Vergleich
    end;

    if p = m+1 then
    begin
      result := t;
      break;
    end;
  end;
end;


function TPatternMatchAnimator.Search_KMP_Count(aShiftArray: TIndexArray): Integer;
var j, k, m, n, i, jold: Integer;
begin
  fCount := 0;
  j := 1;
  //jold := 1;
  k := 1;
  m := Length(fPattern);
  n := Length(fText);
  result := -1;

  while (j <= m) AND (k <= n)  And (Not Terminated) do
  begin
    // Vergleich an aktueller Stelle schlgt fehl
    // ---> Muster entsprechend des Next-Arrays shiften
    while (j > 0) AND (fText[k] <> fPattern[j])  And (Not Terminated) do
    begin
      if UseAnimation then
      begin
        fShift := 0;
        // zunchst: erstmal vergleichen und zeigen,
        // dass der Vergleich wirklich schief geht.
        // zuerst: Den Vergleich zeigen (gelb)
        SetTextMatch(fStartPos, k-1);
        SetPatternMatch(1,j-1);
        SetPatternMismatch(-1,-1);
        SetTextMisMatch(-1,-1);
        SetPatternNextTest(j);
        SetTextNextTest(k);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);

        // jetzt: den Mismatch zeigen
        SetPatternMismatch(j,j);
        SetTextMisMatch(k,k);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;

      // Jetzt: Die Shift bestimmen
      jold := j;
      j := aShiftArray[j];
      inc(fCount);

      // Den Shift animieren
      if UseAnimation then
      begin
        // Muster neu malen - dabei das grn markieren,
        // was man schon wei, den anstehenden Vergleich gelb
        if j = 0 then
        begin
          // j ist gleich 0, d.h. wir werden gleich einmal die Schleife quasi berspringen, ohne was zu vergleichen
          // d.h. wir haben keinen Teil-Match mehr
          SetPatternMatch(-1,-1);
          SetPatternNextTest(j+1);
          // Der Match-Bereich im Muster sind die letzten j Zeichen vor k
          // der Nchste Test ist nochmal k
          SetTextMatch(-1,-1);
          SetTextNextTest(k+1);

          SetPatternMismatch(-1,-1);
          SetTextMisMatch(-1,-1);

          Synchronize(RefreshPattern);
          for i := j to jold-1 do
            ShiftTextNew;
        end
        else
        begin
          // j ist noch grer als 0. D.h. wir berspringen die Schleife gelich NICHT
          // d.h. wir haben noch einen Teil-Match.
          SetPatternMatch(1,j-1);
          SetPatternNextTest(j);
          // Der Match-Bereich im Muster sind die letzten j Zeichen vor k
          // der Nchste Test ist nochmal k
          SetTextMatch(k-j+1,k-1);
          SetTextNextTest(k);

          SetPatternMismatch(-1,-1);
          SetTextMisMatch(-1,-1);

          Synchronize(RefreshPattern);
          for i := j to jold-1 do
            ShiftTextNew;
        end;
      end;
    end;

    // Zeichen stimmt berein - Vergleich animieren
    if (j > 0) AND (fText[k] = fPattern[j])  And (Not Terminated) then
    begin
      inc(fCount);
      if UseAnimation then
      begin
        SetPatternMatch(1,j-1);
        SetPatternNextTest(j);
        SetTextMatch(1,k-1);
        SetTextNextTest(k);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);

        // Vergleich positiv
        SetPatternMatch(1,j);
        SetPatternNextTest(-1);
        SetTextMatch(1,k);
        SetTextNextTest(-1);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
    end;
    inc(k);
    inc(j);
  end;
  if j = m+1 then
    result := k-m;
end;

function TPatternMatchAnimator.Search_BM_Count: Integer;
var m, n, k, j, i: Integer;
  shift, bc, gs: Integer;
begin
  fCount := 0;
  m := Length(fPattern);
  n := Length(fText);

  // k: Position des Musters im Text (aber vom Endes des Musters!)
  k := m;

  result := -1;

  while (k <= n)  And (Not Terminated) do
  begin
    j := 0;
    if UseAnimation then
    begin
      SetPatternMismatch(-1,-1);
      SetTextMismatch(-1,-1);
    end;

    while (j < m) and (fPattern[m-j] = fText[k-j])  And (Not Terminated) do
    begin
      if UseAnimation then
      begin
        // Vergleich animieren
        SetTextMatch(k-j+1, k);
        SetPatternMatch(m-j+1,m);
        SetPatternNextTest(m-j);
        SetTextNextTest(k-j);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
        // einen weiteren Grnen malen
        SetTextMatch(k-j, k);
        SetPatternMatch(m-j,m);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
      inc(j);
      inc(fCount);
    end;

    // den letzten Vergleich zhlen, falls man nicht am Ende ist
    if (j < m) and (fPattern[m-j] <> fText[k-j])  And (Not Terminated) then
    begin
      inc(fCount);
      if UseAnimation then
      begin
        // Vergleich animieren
        SetTextMatch(k-j+1, k);
        SetPatternMatch(m-j+1,m);
        SetPatternNextTest(m-j);
        SetTextNextTest(k-j);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
        // aktuell vergleichener gibt nen Mismatch
        SetTextMismatch(k-j, k-j);
        SetPatternMismatch(m-j,m-j);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
    end;

    if (j = m)  And (Not Terminated) then
    begin
      result := k-j + 1;
      break;
    end else
    begin
      bc := max(1, fBM_BC[Ord(fText[k-j])] - j);
      gs := fBM_GS[m-j];
      shift := max(gs, bc);
      if UseAnimation then
      begin
        // Shift vorbereiten.
        if (bc > gs) or (j = 0)
         then
        begin
          // Wende Bad-Character-Shift an.
          // Grn wird im Text der "BadCharacter", der zum nchsten passenden Char im Muster gemactht wird
          SetTextMatch(k-j, k-j);
          // Dieser Char befindet sich im Muster an Position m-j-shift
          SetPatternMatch(m-j-shift,m-j-shift);

          // Im Text knnen wir keine Mismatches kennen
          SetTextMismatch(-1,-1);
          // Im Pattern zwischen aktueller Position und neuer "GoodChar-Position"
          SetPatternMismatch(m-j-shift+1, m-j-1); // letztes -1 hinzugefgt

          // der Nchste Test-Char im Text ist k+shift, im Muster wieder m
          SetTextNextTest(k+shift);
          SetPatternNextTest(m);

          Synchronize(RefreshPattern);
          Synchronize(RefreshText);
          for i := 1 to shift do
            ShiftTextNew;
        end
        else
        begin
          // Wir mssen den Good-Suffix-Shift anwenden

          // Grn wird im Text das erfolgreich gematchte Suffix
          SetTextMatch(k-j+1, k);
          // Im Muster befindet sich das shift Pltze weiter links
          SetPatternMatch(m-j+1-shift,m-shift);

          // Im Text knnen wir keine Mismatches kennen
          SetTextMismatch(-1,-1);
          // Im Pattern zwischen aktueller Position und neuer "GoodChar-Position"
          SetPatternMismatch(m-shift+1, m-1);

          // der Nchste Test-Char im Text ist k+shift, im Muster wieder m
          SetTextNextTest(k+shift);
          SetPatternNextTest(m);
          Synchronize(RefreshPattern);
          Synchronize(RefreshText);
          for i := 1 to shift do
            ShiftTextNew;
        end;
      end;

      k := k + shift;
    end;
  end;
end;

function TPatternMatchAnimator.Search_Horspool_Count: Integer;
var m, n, k, j, i: Integer;
  shift: Integer;
begin
  fCount := 0;
  m := Length(fPattern);
  n := Length(fText);

  // k: Position des Musters im Text (aber vom Endes des Musters!)
  k := m;

  result := -1;

  while (k <= n)  And (Not Terminated) do
  begin
    j := 0;
    if UseAnimation then
    begin
      SetPatternMismatch(-1,-1);
      SetTextMismatch(-1,-1);
    end;

    while (j < m) and (fPattern[m-j] = fText[k-j])  And (Not Terminated) do
    begin
      if UseAnimation then
      begin
        // Vergleich animieren
        SetTextMatch(k-j+1, k);
        SetPatternMatch(m-j+1,m);
        SetPatternNextTest(m-j);
        SetTextNextTest(k-j);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
        // einen weiteren Grnen malen
        SetTextMatch(k-j, k);
        SetPatternMatch(m-j,m);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
      inc(j);
      inc(fCount);
    end;

    // den letzten Vergleich zhlen, falls man nicht am Ende ist
    if (j < m) and (fPattern[m-j] <> fText[k-j])  And (Not Terminated) then
    begin
      inc(fCount);
      if UseAnimation then
      begin
        // Vergleich animieren
        SetTextMatch(k-j+1, k);
        SetPatternMatch(m-j+1,m);
        SetPatternNextTest(m-j);
        SetTextNextTest(k-j);
        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
        // aktuell verglichener gibt nen Mismatch
        SetTextMismatch(k-j, k-j);
        SetPatternMismatch(m-j,m-j);
        SetPatternNextTest(-1);
        SetTextNextTest(-1);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        SleepAWhile(fDelayTime);
      end;
    end;

    if (j = m)  And (Not Terminated) then
    begin
      result := k-j + 1;
      break;
    end else
    begin

      shift := max(1,fBM_BC[Ord(fText[k])]);
      if UseAnimation then
      begin
        // Shift vorbereiten.

        // Wende Horspool-Bad-Character-Shift an.
        // Grn wird im Text der "BadCharacter", der zum nchsten passenden Char im Muster gemactht wird
        SetTextMatch(k, k);
        // Dieser Char befindet sich im Muster an Position m-shift
        SetPatternMatch(m-shift,m-shift);

        // Im Text knnen wir keine Mismatches kennen
        SetTextMismatch(-1,-1);
        // Im Pattern zwischen aktueller Position und neuer "GoodChar-Position"
        SetPatternMismatch(m-shift+1, m-1); // letztes -1 hinzugefgt

        // der Nchste Test-Char im Text ist k+shift, im Muster wieder m
        SetTextNextTest(k+shift);
        SetPatternNextTest(m);

        Synchronize(RefreshPattern);
        Synchronize(RefreshText);
        for i := 1 to shift do
          ShiftTextNew;
      end;

      k := k + shift;
    end;
  end;
end;


function TPatternMatchAnimator.Search_Count: Integer;
begin
  PreProcessPattern;
  case fMode of
    SA_NAIV     : result := Search_Naiv_Count;
    SA_KMP_F    : result := Search_KMP_Count(fKMP_F);
    SA_KMP_NEXT : result := Search_KMP_Count(fKMP_Next);
    SA_BM,SA_BM_Weak       : result := Search_BM_Count;
    SA_HORSPOOL: result := Search_Horspool_Count;
  else result := -1;
  end;
end;



procedure TPatternMatchAnimator.SetPatternMatch(Start, Ende: Integer);
begin
  fStartGPattern := Start;
  fEndGPattern   := Ende;
end;
procedure TPatternMatchAnimator.SetPatternMismatch(Start, Ende: Integer);
begin
  fStartRPattern := Start;
  fEndRPattern   := Ende;
end;
procedure TPatternMatchAnimator.SetPatternNextTest(Start: Integer);
begin
  fGelbPattern := Start;
end;
procedure TPatternMatchAnimator.SetTextMatch(Start, Ende: Integer);
begin
  fStartGText := Start;
  fEndGText   := Ende;
end;
procedure TPatternMatchAnimator.SetTextMismatch(Start, Ende: Integer);
begin
  fStartRText := Start;
  fEndRText   := Ende;
end;
procedure TPatternMatchAnimator.SetTextNextTest(Start: Integer);
begin
  fGelbText := Start;
end;


procedure TPatternMatchAnimator.RefreshText;
begin
  RepaintTextProc(fMode, fStartPos, fStartGText, fEndGText, fStartRText, fEndRText, fGelbText, fShift);
end;

procedure TPatternMatchAnimator.RefreshPattern;
begin
  RepaintPatternProc(fMode, fStartPos, fStartGPattern, fEndGPattern, fStartRPattern, fEndRPattern, fGelbPattern);
end;

procedure TPatternMatchAnimator.ShiftTextNew;
var s: Integer;
begin
  for s := 1 to ANIMATE_CHARWIDTH do
  begin
    if Terminated then break;
    fShift := s;
    Synchronize(RefreshText);
    SleepAWhile(fDelayTime Div 40);
  end;
  inc(fStartPos);
  fShift := 0;
end;

Procedure TPatternMatchAnimator.SleepAWhile(aWhile: Integer);
begin
  repeat
    sleep(aWhile);
  until (Not fMachePause) or Terminated;
end;


end.
